/*******************************************************************************************************
 *                                                                                                     *
 *        **********                                                                                   *
 *       ************                                                                                  *
 *      ***        ***                                                                                 *
 *      ***   +++   ***                                                                                *
 *      ***   + +   ***                                                                                *
 *      ***   +                               CHIPCON CC2420 BASIC RF LIBRARY                          *
 *      ***   + +   ***                                Initilization                                   *
 *      ***   +++   ***                                                                                *
 *      ***        ***                                                                                 *
 *       ************                                                                                  *
 *        **********                                                                                   *
 *                                                                                                     *
 *******************************************************************************************************
 * This file contains the basic RF initialization function.                                            *
 *                                                                                                     *
 * More information can be found in basic_rf.h                                                         *
 *******************************************************************************************************
 * Compiler: AVR-GCC                                                                                   *
 * Target platform: CC2420DB, CC2420 + any MCU with very few modifications required                    *
 *******************************************************************************************************
 * Revision history:                                                                                   *
 * $Log: basic_rf_init.c,v $
 * Revision 1.4  2004/07/26 11:28:09  mbr
 * Modified RXFIFO flushing by strobing CC2420_SFLUSHRX
 *
 * Revision 1.3  2004/03/30 14:59:22  mbr
 * Release for web
 * 
 *
 *
 *******************************************************************************************************/
#include "include/include.h"


//-------------------------------------------------------------------------------------------------------
// The RF settings structure is declared here, since we'll always need halRfInit()
volatile BASIC_RF_SETTINGS rfSettings;
//-------------------------------------------------------------------------------------------------------




//-------------------------------------------------------------------------------------------------------
//  void basicRfInit(BASIC_RF_RX_INFO *pRRI, UINT8 channel, WORD panId, WORD myAddr)
//
//  DESCRIPTION:
//      Initializes CC2420 for radio communication via the basic RF library functions. Turns on the
//		voltage regulator, resets the CC2420, turns on the crystal oscillator, writes all necessary
//		registers and protocol addresses (for automatic address recognition). Note that the crystal
//		oscillator will remain on (forever).
//
//  ARGUMENTS:
//      BASIC_RF_RX_INFO *pRRI
//          A pointer the BASIC_RF_RX_INFO data structure to be used during the first packet reception.
//			The structure can be switched upon packet reception.
//      UINT8 channel
//          The RF channel to be used (11 = 2405 MHz to 26 = 2480 MHz)
//      WORD panId
//          The personal area network identification number
//      WORD myAddr
//          The 16-bit short address which is used by this node. Must together with the PAN ID form a
//			unique 32-bit identifier to avoid addressing conflicts. Normally, in a 802.15.4 network, the
//			short address will be given to associated nodes by the PAN coordinator.
//-------------------------------------------------------------------------------------------------------
void basicRfInit(BASIC_RF_RX_INFO *pRRI, UINT8 channel, WORD panId, WORD myAddr) {
    UINT8 n;

    // Make sure that the voltage regulator is on, and that the reset pin is inactive
    SET_VREG_ACTIVE();
    halWait(1000);
    SET_RESET_ACTIVE();
    halWait(1000);
    SET_RESET_INACTIVE();
    halWait(500);

    // Initialize the FIFOP external interrupt
    // FIFOP_INT_INIT();
    // ENABLE_FIFOP_INT();

    // Turn off all interrupts while we're accessing the CC2420 registers
	// DISABLE_GLOBAL_INT();

    // Register modifications
    FASTSPI_STROBE(CC2420_SXOSCON);
    //FASTSPI_SETREG(CC2420_MDMCTRL0, 0x0AF2); // Turn on automatic packet acknowledgment
    FASTSPI_SETREG(CC2420_MDMCTRL0, 0x0AE2); // Turn on automatic packet acknowledgment
    FASTSPI_SETREG(CC2420_MDMCTRL1, 0x0500); // Set the correlation threshold = 20
    FASTSPI_SETREG(CC2420_IOCFG0, 0x007F);   // Set the FIFOP threshold to maximum
    FASTSPI_SETREG(CC2420_SECCTRL0, 0x01C4); // Turn off "Security enable"
    
    // Set the RF channel
    halRfSetChannel(channel);

    // Turn interrupts back on
	  ENABLE_GLOBAL_INT();

		// Set the protocol configuration
		rfSettings.pRxInfo = pRRI;
		rfSettings.panId = panId;
		//rfSettings.myAddr = (myAddr+1);//ACK back to some other address.
		rfSettings.myAddr = (myAddr);//ACK back to some other address.
		rfSettings.txSeqNumber = 0;
    rfSettings.receiveOn = FALSE;

		// Wait for the crystal oscillator to become stable
    halRfWaitForCrystalOscillator();

		// Write the short address and the PAN ID to the CC2420 RAM (requires that the XOSC is on and stable)
   	DISABLE_GLOBAL_INT();
    FASTSPI_WRITE_RAM_LE(&myAddr, CC2420RAM_SHORTADDR, 2, n);
    FASTSPI_WRITE_RAM_LE(&panId, CC2420RAM_PANID, 2, n);
  	ENABLE_GLOBAL_INT();

} // basicRfInit
